/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_15;
import java.util.Calendar;
import java.util.Vector;

import djbuch.kapitel_06.GC;
import lotus.domino.*;

/**
 * Kleines DemoObjekt, das die 
 * verschiedenen Szenarien des Recyclings von Domino Objekten 
 * aufzeigt.
 * 
 * @author Thomas Ekert
 *
 */
public class ToyStoreAdvanced {

	private Database db = null;
	private View idSearchView = null, nameSearchView = null, dateSearchView = null;
	private Session session = null;
	private boolean isRecycled = false;
	
	public static final String ITEM_ID = "F_ID";
	public static final String ITEM_NAME = "F_NAME";
	public static final String ITEM_MANUFACTURER = "F_MANUFACTURER";
	public static final String ITEM_CREATIONDATE = "F_CREATIONDATE";
	public static final String ITEM_FORM = "FORM";
	public static final String DEFAULT_FORM = "FO_TOYFORM";	
	public static final String VIEW_TOYBYID = "V_toysByID";
	public static final String VIEW_TOYBYNAME = "V_toysByName";
	public static final String VIEW_TOYBYDATE = "V_toysByDate";
	private static final int APP_ERR_NUM=999;

	/**
	 * Neues ToyStoreObjekt anlegen.
	 * @param dbName - Name der Datenbank. Wird als internes Datebank Objekt angelegt und geffnet
	 * @param session - Um mit Domino Objekten zu arbeiten, wird immer eine Session bentigt.
	 * @throws NotesException
	 */
	public ToyStoreAdvanced (String dbName, Session session) throws NotesException {
		db = session.getDatabase(session.getServerName(),dbName);
		if (db==null || !db.isOpen()) {
			throw new NotesException(APP_ERR_NUM, "Konnte die Datenbank "
					+ dbName + " nicht ffnen.");
		}
		idSearchView = db.getView(VIEW_TOYBYID);
		nameSearchView = db.getView(VIEW_TOYBYNAME);
		dateSearchView = db.getView(VIEW_TOYBYDATE);
		if (idSearchView==null || nameSearchView==null || dateSearchView==null) {
			throw new NotesException(APP_ERR_NUM, "Konnte die Ansicht "
					+ (idSearchView == null ? VIEW_TOYBYID
					: nameSearchView == null ? VIEW_TOYBYNAME
					: dateSearchView == null ? VIEW_TOYBYDATE : "")
					+ " nicht ffnen.");
		}
		this.session = session;
	}

	/**
	 * Ldt den Namen eines Toys mit der ID id
	 * Da das Document doc nur intern in dieser Methode und nur temporr
	 * bentigt wird, kann die Methode selbst das Recycle bernehmen.
	 * @param id - ID des zu ladenden Toys
	 * @return - Name des Toys oder null, falls nicht gefunden.
	 * @throws NotesException
	 */
	public String getToy (int id) throws NotesException {
		checkRecycled();
		Document doc = null;
		try {
			doc = getToyDocument(id);
			return (doc==null?null:doc.getItemValueString(ITEM_NAME));
		} finally {
			GC.recycle (doc);
		}
	}
	
	/**
	 * Ldt das zu einem Toy gehrige Domino Document mit der ID id
	 * Da das Document doc als Ergebnis zurckgegeben wird,
	 * kann die Methode das Recycle nicht (!) selbst bernehmen.
	 * @param id - ID des zu ladenden Toys
	 * @return - Document mit dem Toy oder null, falls nicht gefunden.
	 * @throws NotesException
	 */
	public Document getToyDocument (int id) throws NotesException {
		checkRecycled();
		idSearchView.refresh();
		return idSearchView.getDocumentByKey(new Integer (id));
	}
	
	/**
	 * Erzeugt ein neues Document mit dem Toy
	 * Da das Document doc nur intern in dieser Methode und nur temporr
	 * bentigt wird, kann die Methode selbst das Recycle bernehmen.
	 * @param id - ID des neuen Toys
	 * @param name - Name des neuen Toys
	 * @throws NotesException
	 */
	public int addToy (String name, String manufacturer) throws NotesException {
		checkRecycled();
		Document doc = null;
		try {
			int newID=getNewID();
			doc = db.createDocument();
			doc.replaceItemValue (ITEM_ID, new Integer (newID));
			doc.replaceItemValue (ITEM_NAME, name);
			doc.replaceItemValue (ITEM_MANUFACTURER, manufacturer);
			doc.replaceItemValue (ITEM_FORM, DEFAULT_FORM);
			doc.replaceItemValue (ITEM_CREATIONDATE, session.createDateTime(Calendar.getInstance()));
			doc.save();
			return newID;
		} finally {
			GC.recycle (doc);
		}
	}
	/**
	 * @see findToyByName (String, String)
	 * @param name
	 * @return
	 * @throws NotesException
	 */
	public Document findToyByName (String name) throws NotesException {
		return findToyByName (name, null);
	}
	/**
	 * Schlgt in der Ansicht V_toysByName den Namen in der ersten Spalte
	 * und optional den Hersteller in der zweiten Spalte nach.
	 * Der erste Treffer wird zurckgegeben.
	 * @param name
	 * @param manufacturer
	 * @return
	 * @throws NotesException
	 */
	public Document findToyByName (String name, String manufacturer) throws NotesException {
		checkRecycled();
		Vector searchKeys = new Vector ();
		searchKeys.add(name);
		if (manufacturer!=null) {
			searchKeys.add(manufacturer);
		}
		nameSearchView.refresh();
		return nameSearchView.getDocumentByKey(searchKeys,false);
	}
	
	/**
	 * @see findAllToysByName (String, String)
	 * @param name
	 * @return
	 * @throws NotesException
	 */
	public DocumentCollection findAllToysByName (String name) throws NotesException {
		return findAllToysByName (name, null);
	}
	/**
	 * Schlgt in der Ansicht V_toysByName den Namen in der ersten Spalte
	 * und optional den Hersteller in der zweiten Spalte nach.
	 * ALLE Treffer werden zurckgegeben.
	 * @param name
	 * @param manufacturer
	 * @return
	 * @throws NotesException
	 */
	public DocumentCollection findAllToysByName (String name, String manufacturer) throws NotesException {
		checkRecycled();
		Vector searchKeys = new Vector ();
		searchKeys.add(name);
		if (manufacturer!=null) {
			searchKeys.add(manufacturer);			
		}
		nameSearchView.refresh();
		return nameSearchView.getAllDocumentsByKey(searchKeys,false);
	}
	
	/**
	 * Gibt alle Toys zurck, die am gleichen Datum erstellt wurden,
	 * wie der Datumsanteil von dt
	 * Der Datumsanteil wird anhand einer Kopie des dt Objekts berechnet,
	 * um keine Nebenwirkungen auf dt zu haben.
	 * @param dt
	 * @return
	 * @throws NotesException
	 */
	public DocumentCollection findAllToysByDate (DateTime dt) throws NotesException {
		checkRecycled();
		DateTime datum = null;
		try {
			datum = session.createDateTime(dt.toJavaDate());
			dateSearchView.refresh();
			datum.setAnyTime(); //nur Datum berprfen
			return dateSearchView.getAllDocumentsByKey(datum);
		} finally {
			GC.recycle (datum);
		}
	}
	
	/**
	 * Fhrt eine Volltextsuche im View V_toysByID durch
	 * Einschrnkend wirkt hier nur die SELECT Formel der Ansicht, 
	 * d.h. es werden nur Dokumente durchsucht, die im View angezeigt werden.
	 * Dennoch werden alle (!) Daten jedes Dokuments durchsucht, auch wenn
	 * sie nicht im View angezeigt werden (im Gegensatz zu findAllToysByName etc.)
	 * @param searchQuery
	 * @return
	 * @throws NotesException
	 */
	public ViewEntryCollection findAllFT (String searchQuery) throws NotesException {
		checkRecycled();
		if (!db.isFTIndexed()) {
			System.out.println ("[!!!] WARNUNG: Datenbank ist nicht FT indiziert.");
		}
		try {
			idSearchView.refresh();
			idSearchView.FTSearch(searchQuery);
			return idSearchView.getAllEntries();
		} finally {
			idSearchView.clear();
		}
	}
	
	/**
	 * ldt im nach ID aufsteigend sortierten View idSearchView den
	 * letzten Eintrag, also die hchste Verfgbare ID und gibt einen
	 * um 1 greren Wert zurck.
	 * Hier soll die Technik des Nachschlagens gezeigt werden.
	 * Artikelnummern sollten in einer realen Anwendung besser einer eigenen Klasse
	 * und einem eigenen Algorithmus generiert werden.
	 * @return
	 * @throws NotesException
	 */
	private int getNewID () throws NotesException {
		Document doc = null;
		try {
			synchronized (idSearchView) {
				idSearchView.refresh();
				doc = idSearchView.getLastDocument();
				if (doc==null) {
					return 1;
				} else {
					return doc.getItemValueInteger(ITEM_ID) + 1;
				}
			}
		} finally {
			GC.recycle(doc);
		}
	}
	
	/**
	 * Lscht das Document des Toy mit der ID id.
	 * Da das Document doc nur intern in dieser Methode und nur temporr
	 * bentigt wird, kann die Methode selbst das Recycle bernehmen.
 	 * @param id
	 * @throws NotesException
	 */
	public void removeToy (int id) throws NotesException {
		checkRecycled();
		Document doc = null;
		try {
			doc = this.getToyDocument(id);
			if (doc!=null) {
				doc.remove(true);
			}
		} finally {
			GC.recycle(doc);
		}
	}
	
	/**
	 * Da die Klasse Toy eine Instanzvariable Database db
	 * selbst als Objekt bezieht und vorhlt, muss eine
	 * externe Mglichkeit geboten werden, dieses Domino Java
	 * Objekt dem Recycle zuzufhren.
	 * Gleichzeitig wird ber diese Methode die
	 * Instanzvariable db auf null gesetzt, um in checkRecycled
	 * erkennen zu knnen, ob ein Recycling erfolgt war.
	 */
	public void recycle () {
		GC.recycle (idSearchView);
		GC.recycle (nameSearchView);
		GC.recycle (dateSearchView);
		GC.recycle(db);
		isRecycled=true;
	}
	
	/**
	 * Prft, ob das ToyStore Objekt noch gltig ist.
	 * @throws NotesException
	 */
	private void checkRecycled () throws NotesException {
		if (isRecycled) {
			throw new NotesException(NotesError.NOTES_ERR_DELETED,
					"Object has been removed or recycled");
		}
	}
	
	public DocumentCollection search (String query) throws NotesException {
		return db.search(query);
	}
}
